Rem AddGuides.vbs
Rem An InDesign VBScript
Rem
Rem Adds guides around the selected object or objects.
Rem
Rem Choose Visible Bounds to consider the stroke weight of the paths when placing the guides,
Rem or choose Geometric Bounds to ignore the stroke weight when placing the guides
Rem Choose Each Item to position the guides around individual items in the selection,
Rem or Entire Selection to position the guides around the selection as a whole.
Rem
Rem For more on InDesign/InCopy scripting see the documentation included in the Scripting SDK 
Rem available at http://www.adobe.com/devnet/indesign/sdk.html
Rem or visit the InDesign scripting user to user forum at http://www.adobeforums.com.
Rem
main
Function main()
	ReDim myObjectList(0)
	Set myInDesign = CreateObject("InDesign.Application.2024")
	Rem Set the user interaction level to allow the display of dialog boxes and alerts.
	myInDesign.ScriptPreferences.UserInteractionLevel = idUserInteractionLevels.idInteractWithAll
	If myInDesign.Documents.Count <> 0 Then
	    If myInDesign.Selection.Count <> 0 Then
	        For myCounter = 1 To myInDesign.Selection.Count
	            Select Case TypeName(myInDesign.Selection.Item(myCounter))
	                Case "Rectangle", "Oval", "Polygon", "GraphicLine", "Button", "Group", "TextFrame"
	                    If Not (IsEmpty(myObjectList(0))) Then
	                        ReDim Preserve myObjectList(UBound(myObjectList) + 1)
	                    End If
	                    Set myObjectList(UBound(myObjectList)) = myInDesign.Selection.Item(myCounter)
	            End Select
	        Next
	        If Not (IsEmpty(myObjectList(0))) Then
	            myDisplayDialog myInDesign, myObjectList
	        End If
	    Else
	        MsgBox ("Nothing is selected. Please select an object and try again.")
	    End If
	Else
	    MsgBox ("Please open a document, select an object, and try again.")
	End If
End Function
Function myDisplayDialog(myInDesign, myObjectList)
Set myDialog = myInDesign.Dialogs.Add
myDialog.Name = "AddGuides"
With myDialog.DialogColumns.Add
    With .BorderPanels.Add
            With .DialogColumns.Add
                With .StaticTexts.Add
                    .StaticLabel = "Add Guides at:"
                End With
            End With
            With .DialogColumns.Add
                Set myTopCheckbox = .CheckboxControls.Add
                myTopCheckbox.StaticLabel = "&Top"
                myTopCheckbox.CheckedState = True
                Set myLeftCheckbox = .CheckboxControls.Add
                myLeftCheckbox.StaticLabel = "&Left"
                myLeftCheckbox.CheckedState = True
                Set myBottomCheckbox = .CheckboxControls.Add
                myBottomCheckbox.StaticLabel = "&Bottom"
                myBottomCheckbox.CheckedState = True
                Set myRightCheckbox = .CheckboxControls.Add
                myRightCheckbox.StaticLabel = "&Right"
                myRightCheckbox.CheckedState = True
                Set myXCenterCheckbox = .CheckboxControls.Add
                myXCenterCheckbox.StaticLabel = "&Horizontal Center"
                myXCenterCheckbox.CheckedState = True
                Set myYCenterCheckbox = .CheckboxControls.Add
                myYCenterCheckbox.StaticLabel = "&Vertical Center"
                myYCenterCheckbox.CheckedState = True
                Set myXPointCheckbox = .CheckboxControls.Add
                myXPointCheckbox.StaticLabel = "Path Point Hori&zontal Center"
                myXPointCheckbox.CheckedState = False
                Set myYPointCheckbox = .CheckboxControls.Add
                myYPointCheckbox.StaticLabel = "Path Point Verti&cal Center"
                myYPointCheckbox.CheckedState = False
            End With
    End With
    With .BorderPanels.Add
            With .DialogColumns.Add
                With .StaticTexts.Add
                    .StaticLabel = "Add Guides Around:"
                End With
            End With
            With .DialogColumns.Add
                Set myRangeButtons = .RadiobuttonGroups.Add
                With myRangeButtons
                    With .RadiobuttonControls.Add
                        .StaticLabel = "Each &Object"
                        .CheckedState = True
                    End With
                    With .RadiobuttonControls.Add
                        .StaticLabel = "Entire &Selection"
                    End With
                End With
            End With
    End With
    With .BorderPanels.Add
            With .DialogColumns.Add
                With .StaticTexts.Add
                    .StaticLabel = "Guides Based On:"
                End With
            End With
            With .DialogColumns.Add
                Set myBasedOnButtons = .RadiobuttonGroups.Add
                With myBasedOnButtons
                    With .RadiobuttonControls.Add
                        .StaticLabel = "&Geometric Bounds"
                        .CheckedState = True
                    End With
                    With .RadiobuttonControls.Add
                        .StaticLabel = "V&isible Bounds"
                    End With
                End With
            End With
    End With
End With
myReturn = myDialog.Show
If myReturn = True Then
    Rem Get the values from the dialog box.
    myTop = myTopCheckbox.CheckedState
    myLeft = myLeftCheckbox.CheckedState
    myBottom = myBottomCheckbox.CheckedState
    myRight = myRightCheckbox.CheckedState
    myXCenter = myXCenterCheckbox.CheckedState
    myYCenter = myYCenterCheckbox.CheckedState
    myXPoint = myXPointCheckbox.CheckedState
    myYPoint = myYPointCheckbox.CheckedState
    myRange = myRangeButtons.SelectedButton
    myBasedOn = myBasedOnButtons.SelectedButton
    myDialog.Destroy
    If (myTop + myLeft + myBottom + myRight + myXCenter + myYCenter + myXPoint + myYPoint) <> 0 Then
        myAddGuides myInDesign, myTop, myLeft, myBottom, myRight, myXCenter, myYCenter, myRange, myBasedOn, myXPoint, myYPoint, myObjectList
    End If
End If
End Function
Function myAddGuides(myInDesign, myTop, myLeft, myBottom, myRight, myXCenter, myYCenter, myRange, myBasedOn, myXPoint, myYPoint, myObjectList)
    Set myDocument = myInDesign.ActiveDocument
    myOldRulerOrigin = myDocument.ViewPreferences.RulerOrigin
    myDocument.ViewPreferences.RulerOrigin = idRulerOrigin.idSpineOrigin
    Rem Create a layer to hold the guides (if it does not already exist).
    Err.Clear
    On Error Resume Next
    Set myLayer = myDocument.Layers.Item("Guides")
    If Err.Number <> 0 Then
        Set myLayer = myDocument.Layers.Add
        myLayer.Name = "Guides"
        Err.Clear
    End If
    On Error GoTo 0
    'Set mySpread = myInDesign.ActiveWindow.ActiveSpread
    Rem Process the objects in myObjectList.
    For myCounter = 0 To UBound(myObjectList)
        Set myObject = myObjectList(myCounter)
        If myXPoint = True Or myYPoint = True Then
        	myDrawGuidesAtPathPoints myObject, myXPoint, myYPoint, myLayer
        End If
        Rem Use geometric bounds or visible bounds.
        If myBasedOn = 0 Then
            myBounds = myObject.GeometricBounds
        Else
            myBounds = myObject.VisibleBounds
        End If
        Rem If we're at the first object, set up some initial bounding box values.
        If myCounter = 0 Then
            myX1 = myBounds(1)
            myY1 = myBounds(0)
            myX2 = myBounds(3)
            myY2 = myBounds(2)
        End If
        If myRange = 0 Then
            If TypeName(myObject.Parent) = "Page" Or TypeName(myObject.Parent) = "Spread" Then
                Set myPage = myObject.Parent
            Else
                Set myPage = myInDesign.ActiveWindow.ActiveSpread
            End If
            myDrawGuides myPage, myBounds(1), myBounds(0), myBounds(3), myBounds(2), myTop, myLeft, myBottom, myRight, myXCenter, myYCenter, myLayer
        Else
            Rem Compare the bounds values to the stored bounds.
            Rem If a given bounds value is less than (for x1 and y1) or
            Rem greater than (for x2 and y2) the stored value,
            Rem then replace the stored value with the bounds value.
            If myBounds(0) < myY1 Then
                myY1 = myBounds(0)
            End If
            If myBounds(1) < myX1 Then
                myX1 = myBounds(1)
            End If
            If myBounds(2) > myY2 Then
                myY2 = myBounds(2)
            End If
            If myBounds(3) > myX2 Then
                myX2 = myBounds(3)
            End If
        End If
    Next
    If myRange <> 0 Then
        If ((TypeName(myObject.Parent) = "Page") Or (TypeName(myObject.Parent) = "Spread")) Then
            Set myPage = myObject.Parent
        Else
            Set myPage = myInDesign.ActiveWindow.ActiveSpread
        End If
        myDrawGuides myPage, myX1, myY1, myX2, myY2, myTop, myLeft, myBottom, myRight, myXCenter, myYCenter, myLayer
    End If
    myDocument.ViewPreferences.RulerOrigin = myOldRulerOrigin
End Function
Function myDrawGuidesAtPathPoints(myObject, myXPoint, myYPoint, myLayer)
	For myPathCounter = 1 To myObject.Paths.Count
		for myPointCounter = 1 To myObject.Paths.Item(myPathCounter).PathPoints.Count
			If myXPoint = True Then
				myDrawGuide myObject.parent, myObject.Paths.Item(myPathCounter).PathPoints.Item(myPointCounter).Anchor(0), 1, myLayer
			End If
			If myYPoint = True Then
				myDrawGuide myObject.parent, myObject.Paths.Item(myPathCounter).PathPoints.Item(myPointCounter).Anchor(1), 0, myLayer
			End If
		Next
	Next
End Function
Function myDrawGuides(myPage, myX1, myY1, myX2, myY2, myTop, myLeft, myBottom, myRight, myXCenter, myYCenter, myLayer)
    If myTop = True Then
        myDrawGuide myPage, myY1, 0, myLayer
    End If
    If myLeft = True Then
        myDrawGuide myPage, myX1, 1, myLayer
    End If
    If myBottom = True Then
        myDrawGuide myPage, myY2, 0, myLayer
    End If
    If myRight = True Then
        myDrawGuide myPage, myX2, 1, myLayer
    End If
    If myXCenter = True Then
        myDrawGuide myPage, myX1 + ((myX2 - myX1) / 2), 1, myLayer
    End If
    If myYCenter = True Then
        myDrawGuide myPage, myY1 + ((myY2 - myY1) / 2), 0, myLayer
    End If
End Function
Function myDrawGuide(myPage, myLocation, myGuideOrientation, myLayer)
    Set myGuide = myPage.Guides.Add(myLayer)
    If myGuideOrientation = 0 Then
        myGuide.Orientation = idHorizontalOrVertical.idHorizontal
    Else
        myGuide.Orientation = idHorizontalOrVertical.idVertical
    End If
    myGuide.Location = myLocation
End Function

'' SIG '' Begin signature block
'' SIG '' MIIpyQYJKoZIhvcNAQcCoIIpujCCKbYCAQExDzANBglg
'' SIG '' hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
'' SIG '' BgEEAYI3AgEeMCQCAQEEEE7wKRaZJ7VNj+Ws4Q8X66sC
'' SIG '' AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
'' SIG '' wwd/fRjWugoX8vLHHpDMc1UWLjWsLLPJQcmNQMq49nyg
'' SIG '' gg6WMIIGsDCCBJigAwIBAgIQCK1AsmDSnEyfXs2pvZOu
'' SIG '' 2TANBgkqhkiG9w0BAQwFADBiMQswCQYDVQQGEwJVUzEV
'' SIG '' MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
'' SIG '' d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2Vy
'' SIG '' dCBUcnVzdGVkIFJvb3QgRzQwHhcNMjEwNDI5MDAwMDAw
'' SIG '' WhcNMzYwNDI4MjM1OTU5WjBpMQswCQYDVQQGEwJVUzEX
'' SIG '' MBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMT
'' SIG '' OERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5n
'' SIG '' IFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMIICIjANBgkq
'' SIG '' hkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA1bQvQtAorXi3
'' SIG '' XdU5WRuxiEL1M4zrPYGXcMW7xIUmMJ+kjmjYXPXrNCQH
'' SIG '' 4UtP03hD9BfXHtr50tVnGlJPDqFX/IiZwZHMgQM+TXAk
'' SIG '' ZLON4gh9NH1MgFcSa0OamfLFOx/y78tHWhOmTLMBICXz
'' SIG '' ENOLsvsI8IrgnQnAZaf6mIBJNYc9URnokCF4RS6hnyzh
'' SIG '' GMIazMXuk0lwQjKP+8bqHPNlaJGiTUyCEUhSaN4QvRRX
'' SIG '' XegYE2XFf7JPhSxIpFaENdb5LpyqABXRN/4aBpTCfMjq
'' SIG '' GzLmysL0p6MDDnSlrzm2q2AS4+jWufcx4dyt5Big2MEj
'' SIG '' R0ezoQ9uo6ttmAaDG7dqZy3SvUQakhCBj7A7CdfHmzJa
'' SIG '' wv9qYFSLScGT7eG0XOBv6yb5jNWy+TgQ5urOkfW+0/tv
'' SIG '' k2E0XLyTRSiDNipmKF+wc86LJiUGsoPUXPYVGUztYuBe
'' SIG '' M/Lo6OwKp7ADK5GyNnm+960IHnWmZcy740hQ83eRGv7b
'' SIG '' UKJGyGFYmPV8AhY8gyitOYbs1LcNU9D4R+Z1MI3sMJN2
'' SIG '' FKZbS110YU0/EpF23r9Yy3IQKUHw1cVtJnZoEUETWJrc
'' SIG '' JisB9IlNWdt4z4FKPkBHX8mBUHOFECMhWWCKZFTBzCEa
'' SIG '' 6DgZfGYczXg4RTCZT/9jT0y7qg0IU0F8WD1Hs/q27Iwy
'' SIG '' CQLMbDwMVhECAwEAAaOCAVkwggFVMBIGA1UdEwEB/wQI
'' SIG '' MAYBAf8CAQAwHQYDVR0OBBYEFGg34Ou2O/hfEYb7/mF7
'' SIG '' CIhl9E5CMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/n
'' SIG '' upiuHA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAK
'' SIG '' BggrBgEFBQcDAzB3BggrBgEFBQcBAQRrMGkwJAYIKwYB
'' SIG '' BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBB
'' SIG '' BggrBgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQw
'' SIG '' QwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGln
'' SIG '' aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5j
'' SIG '' cmwwHAYDVR0gBBUwEzAHBgVngQwBAzAIBgZngQwBBAEw
'' SIG '' DQYJKoZIhvcNAQEMBQADggIBADojRD2NCHbuj7w6mdNW
'' SIG '' 4AIapfhINPMstuZ0ZveUcrEAyq9sMCcTEp6QRJ9L/Z6j
'' SIG '' fCbVN7w6XUhtldU/SfQnuxaBRVD9nL22heB2fjdxyyL3
'' SIG '' WqqQz/WTauPrINHVUHmImoqKwba9oUgYftzYgBoRGRjN
'' SIG '' YZmBVvbJ43bnxOQbX0P4PpT/djk9ntSZz0rdKOtfJqGV
'' SIG '' WEjVGv7XJz/9kNF2ht0csGBc8w2o7uCJob054ThO2m67
'' SIG '' Np375SFTWsPK6Wrxoj7bQ7gzyE84FJKZ9d3OVG3ZXQIU
'' SIG '' H0AzfAPilbLCIXVzUstG2MQ0HKKlS43Nb3Y3LIU/Gs4m
'' SIG '' 6Ri+kAewQ3+ViCCCcPDMyu/9KTVcH4k4Vfc3iosJocsL
'' SIG '' 6TEa/y4ZXDlx4b6cpwoG1iZnt5LmTl/eeqxJzy6kdJKt
'' SIG '' 2zyknIYf48FWGysj/4+16oh7cGvmoLr9Oj9FpsToFpFS
'' SIG '' i0HASIRLlk2rREDjjfAVKM7t8RhWByovEMQMCGQ8M4+u
'' SIG '' KIw8y4+ICw2/O/TOHnuO77Xry7fwdxPm5yg/rBKupS8i
'' SIG '' bEH5glwVZsxsDsrFhsP2JjMMB0ug0wcCampAMEhLNKhR
'' SIG '' ILutG4UI4lkNbcoFUCvqShyepf2gpx8GdOfy1lKQ/a+F
'' SIG '' SCH5Vzu0nAPthkX0tGFuv2jiJmCG6sivqf6UHedjGzqG
'' SIG '' VnhOMIIH3jCCBcagAwIBAgIQD99ExMbwCe2qSx8hp1o/
'' SIG '' EDANBgkqhkiG9w0BAQsFADBpMQswCQYDVQQGEwJVUzEX
'' SIG '' MBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMT
'' SIG '' OERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5n
'' SIG '' IFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMB4XDTIzMTEw
'' SIG '' MzAwMDAwMFoXDTI1MTEwNDIzNTk1OVowgeYxEzARBgsr
'' SIG '' BgEEAYI3PAIBAxMCVVMxGTAXBgsrBgEEAYI3PAIBAhMI
'' SIG '' RGVsYXdhcmUxHTAbBgNVBA8MFFByaXZhdGUgT3JnYW5p
'' SIG '' emF0aW9uMRAwDgYDVQQFEwcyNzQ4MTI5MQswCQYDVQQG
'' SIG '' EwJVUzELMAkGA1UECBMCY2ExETAPBgNVBAcTCFNhbiBK
'' SIG '' b3NlMRMwEQYDVQQKEwpBZG9iZSBJbmMuMSwwKgYDVQQL
'' SIG '' EyNJbGx1c3RyYXRvciwgSW5EZXNpZ24sIEluQ29weSwg
'' SIG '' TXVzZTETMBEGA1UEAxMKQWRvYmUgSW5jLjCCAiIwDQYJ
'' SIG '' KoZIhvcNAQEBBQADggIPADCCAgoCggIBAPCR5c/cgyg4
'' SIG '' MpJD46/+BWENPhE1RjN77VE2BWlIUdiO2nlWelKthEq9
'' SIG '' aK+1ofKIlDVc9mvxksuXBLESi7HBt3PCVBntNJkjBJd0
'' SIG '' EXeprmDK8HZrz8y/KSwEewbTMjneMRIRcJV2GMoAyUzK
'' SIG '' EyFJF7m6ByeoNBDDXMD4AQF5C2l3wfvRbREw+G+cyeya
'' SIG '' 564AHpFJ9sfqi9jAycnEZISxEf6N72GgGQr04cZt3JTb
'' SIG '' TktZGt6uWdZnseXDLGRHzFu4P4EqmVKvfMDliaKyWBzV
'' SIG '' lY+gmtE1vSxIK79WTySFrFY5j1vtbKyHybk6tv4TOmLt
'' SIG '' Lw4OP/ngVDiw/ZP+b7saGxPOEfuh5LOAr+llt+6V5NhT
'' SIG '' dkYPR6cbokawwX2DbGg7IeVuUPdf9bx/LznFV/8kPxCB
'' SIG '' J6qg0dqvCmQt/XbBXPJUvDoVJ9z+9hDlrAoLESqpHK8U
'' SIG '' 73UFGohBMJ/7itIyKngem7Ms98z6x0i63FJA8C06g+XE
'' SIG '' GJUmR28+c6ba+m/sc6xHI4zdL5a84UeRS9+u8UL8N4av
'' SIG '' fpSxHYTTxVjk1ZlwQ8dLARwEVWvTu4lRcEck3RPogx7o
'' SIG '' o8wUpSw9++m+Oi+K/z7RP/R67uCEtrkdM5N/xUoQR/t6
'' SIG '' 3P2ia9g8+LuYJ2BNBd1+qMcSoSlDfIFLFMeeqESbMv+j
'' SIG '' UsRQJn2lZvufAgMBAAGjggICMIIB/jAfBgNVHSMEGDAW
'' SIG '' gBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4EFgQU
'' SIG '' RPnrJ9yT26rMrd/Q4PfQSlgeaRcwPQYDVR0gBDYwNDAy
'' SIG '' BgVngQwBAzApMCcGCCsGAQUFBwIBFhtodHRwOi8vd3d3
'' SIG '' LmRpZ2ljZXJ0LmNvbS9DUFMwDgYDVR0PAQH/BAQDAgeA
'' SIG '' MBMGA1UdJQQMMAoGCCsGAQUFBwMDMIG1BgNVHR8Ega0w
'' SIG '' gaowU6BRoE+GTWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNv
'' SIG '' bS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWduaW5nUlNB
'' SIG '' NDA5NlNIQTM4NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRw
'' SIG '' Oi8vY3JsNC5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVz
'' SIG '' dGVkRzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIx
'' SIG '' Q0ExLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQwJAYIKwYB
'' SIG '' BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBc
'' SIG '' BggrBgEFBQcwAoZQaHR0cDovL2NhY2VydHMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25p
'' SIG '' bmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcnQwCQYDVR0T
'' SIG '' BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAPIZ9C4IuKg85
'' SIG '' 0H8M66OLMVllIpANmEjijIG/D65qP6jfma+2WFps2bb+
'' SIG '' vKKIEtG9TFtCAlSrA1OV6Ru78paq0gqw/q3QAN0OleFR
'' SIG '' F1NvT/TLO0ulhjXTETB1BwSKZaJH001DH0+60TmbGfVh
'' SIG '' mwLrEj2Zfl4COFwivZ/2tRoknZl5tmiu4hpjDbOapWd9
'' SIG '' 5cqliSoMeu2i588557eikdiDk6bXBa5MxHjyXvbvaIWN
'' SIG '' Zjsymn0L0sYkF3BToDN0h0XiTTZjZPHdSV2GjX6C/+Nj
'' SIG '' HSZGRtBaIxdAL1mKEWJV9qkEKELPb098rO+zGdn1E5IH
'' SIG '' pS1zSRIYngwZEjZfyoCkg1F+EW4QPs2+ff2CsW5i3IEd
'' SIG '' 7vL19w9oyce3S4wejvnlTKt+EQWn3gSPxrbkp3BxdWRN
'' SIG '' 25MKo5bS5bJ21BiwqcIJHcTxWNgaQPh4sZboi9IimCSn
'' SIG '' UenPQhqdeOuz0NnmepMEYd7L/kqnAeUZ8bW1UbITAWQ/
'' SIG '' UT3phjCfRQNpEh4OBoUcH8hOOHZ0Lh86mxv9lDP3i7c/
'' SIG '' bQzZnrJr2dS34JXolwREbTX9Yu5JB5X3YP539/M1Ms5O
'' SIG '' FZou/MHQBrWMsVBsEXDTWqwbMTYw/zvIiHXcGUnpzInw
'' SIG '' 1jQXlLrmwVHW4ug+A4LFO7bWC4mnZoDNWp0xnlGmbP6I
'' SIG '' ccJmW5C7RJMxghqLMIIahwIBATB9MGkxCzAJBgNVBAYT
'' SIG '' AlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFBMD8G
'' SIG '' A1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNp
'' SIG '' Z25pbmcgUlNBNDA5NiBTSEEzODQgMjAyMSBDQTECEA/f
'' SIG '' RMTG8AntqksfIadaPxAwDQYJYIZIAWUDBAIBBQCggZww
'' SIG '' GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYB
'' SIG '' BAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcN
'' SIG '' AQkEMSIEIKH340VGvKgLYH7pAbpeZOqmRAahEqxLe/mV
'' SIG '' hsoh0W+LMDAGCisGAQQBgjcCAQwxIjAgoB6AHABBAGQA
'' SIG '' bwBiAGUAIABJAG4ARABlAHMAaQBnAG4wDQYJKoZIhvcN
'' SIG '' AQEBBQAEggIAINqMYiFeFPAUHZBKbJuXEQ8HE4aqg60Q
'' SIG '' YCn8wZVPfDXjSmEeZZB2s80vAuJHbwjugxWmIBXx7wTO
'' SIG '' gcNBaSb9CKhGoO4cciqNOphPaNzMs/ldzb3maKLzi3Yy
'' SIG '' Q0ZPPyp+CCrgYFMdpfnNv+HoVhl6l7r2KC5PkS9UuPhu
'' SIG '' 1o3BtQQ8zBd5/TNG6uT1hQKvCzT5pti6G1GXanYFZmFe
'' SIG '' Y35dC1gdzD+nNCzNuNPvTBpKxaeDE5n7yJ5l5nzVXZlx
'' SIG '' ViZcu3PmZiik2CXz/xWpwBYp3tjSy0w/LoQx7bU4/+Fy
'' SIG '' 6r9ixWDXmlpWQZbBGtSZjoGhA2l0OAzcg5FjBEHOvlQD
'' SIG '' BJC+UIQJd7AC1xf6T/WZDBSjJnlOiPfB4f71/9naS8D4
'' SIG '' GPYbLzQUigBlfLbEnQY9cuTe0CMqr6Jt/8Dq51EsBLyf
'' SIG '' uBVlV4OKdhlWdkiCoWSy9A2CKASUO2gsmqhHYwhEPXm2
'' SIG '' KwMYu3SxhID5y6tlI8UAOi2sGaX+gbROdhA7vXlXqKtg
'' SIG '' fHnWc7VB0lwyjkaQ1LDdR9H4USRS3tQnbu7aw3gmhTIO
'' SIG '' EpifqYUDbO7wl9CvqRgEIr6zg9lTFgY9/f2ujWYYRT51
'' SIG '' O+3d/R8n9cvuWV/lcWdZORwpDob7+RNx0KGGaISfj4Ee
'' SIG '' iR4wi4Bfmrw357ygBy5O8ZglGne/vCANQQahghdAMIIX
'' SIG '' PAYKKwYBBAGCNwMDATGCFywwghcoBgkqhkiG9w0BBwKg
'' SIG '' ghcZMIIXFQIBAzEPMA0GCWCGSAFlAwQCAQUAMHgGCyqG
'' SIG '' SIb3DQEJEAEEoGkEZzBlAgEBBglghkgBhv1sBwEwMTAN
'' SIG '' BglghkgBZQMEAgEFAAQgj6kefqKfSkTwZkSZn3N1/w6q
'' SIG '' TLFP6eIuWKnf1pYhijsCEQCTtfrZIk5qumxvl9Da02xm
'' SIG '' GA8yMDI0MDEyNjIzNTQ0MlqgghMJMIIGwjCCBKqgAwIB
'' SIG '' AgIQBUSv85SdCDmmv9s/X+VhFjANBgkqhkiG9w0BAQsF
'' SIG '' ADBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNl
'' SIG '' cnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0
'' SIG '' ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5n
'' SIG '' IENBMB4XDTIzMDcxNDAwMDAwMFoXDTM0MTAxMzIzNTk1
'' SIG '' OVowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lD
'' SIG '' ZXJ0LCBJbmMuMSAwHgYDVQQDExdEaWdpQ2VydCBUaW1l
'' SIG '' c3RhbXAgMjAyMzCCAiIwDQYJKoZIhvcNAQEBBQADggIP
'' SIG '' ADCCAgoCggIBAKNTRYcdg45brD5UsyPgz5/X5dLnXaEO
'' SIG '' CdwvSKOXejsqnGfcYhVYwamTEafNqrJq3RApih5iY2nT
'' SIG '' WJw1cb86l+uUUI8cIOrHmjsvlmbjaedp/lvD1isgHMGX
'' SIG '' lLSlUIHyz8sHpjBoyoNC2vx/CSSUpIIa2mq62DvKXd4Z
'' SIG '' GIX7ReoNYWyd/nFexAaaPPDFLnkPG2ZS48jWPl/aQ9OE
'' SIG '' 9dDH9kgtXkV1lnX+3RChG4PBuOZSlbVH13gpOWvgeFmX
'' SIG '' 40QrStWVzu8IF+qCZE3/I+PKhu60pCFkcOvV5aDaY7Mu
'' SIG '' 6QXuqvYk9R28mxyyt1/f8O52fTGZZUdVnUokL6wrl76f
'' SIG '' 5P17cz4y7lI0+9S769SgLDSb495uZBkHNwGRDxy1Uc2q
'' SIG '' TGaDiGhiu7xBG3gZbeTZD+BYQfvYsSzhUa+0rRUGFOpi
'' SIG '' CBPTaR58ZE2dD9/O0V6MqqtQFcmzyrzXxDtoRKOlO0L9
'' SIG '' c33u3Qr/eTQQfqZcClhMAD6FaXXHg2TWdc2PEnZWpST6
'' SIG '' 18RrIbroHzSYLzrqawGw9/sqhux7UjipmAmhcbJsca8+
'' SIG '' uG+W1eEQE/5hRwqM/vC2x9XH3mwk8L9CgsqgcT2ckpME
'' SIG '' tGlwJw1Pt7U20clfCKRwo+wK8REuZODLIivK8SgTIUlR
'' SIG '' fgZm0zu++uuRONhRB8qUt+JQofM604qDy0B7AgMBAAGj
'' SIG '' ggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/
'' SIG '' BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAgBgNV
'' SIG '' HSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwHwYD
'' SIG '' VR0jBBgwFoAUuhbZbU2FL3MpdpovdYxqII+eyG8wHQYD
'' SIG '' VR0OBBYEFKW27xPn783QZKHVVqllMaPe1eNJMFoGA1Ud
'' SIG '' HwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
'' SIG '' LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEy
'' SIG '' NTZUaW1lU3RhbXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEB
'' SIG '' BIGDMIGAMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
'' SIG '' aWdpY2VydC5jb20wWAYIKwYBBQUHMAKGTGh0dHA6Ly9j
'' SIG '' YWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0
'' SIG '' ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5j
'' SIG '' cnQwDQYJKoZIhvcNAQELBQADggIBAIEa1t6gqbWYF7xw
'' SIG '' jU+KPGic2CX/yyzkzepdIpLsjCICqbjPgKjZ5+PF7SaC
'' SIG '' inEvGN1Ott5s1+FgnCvt7T1IjrhrunxdvcJhN2hJd6Pr
'' SIG '' kKoS1yeF844ektrCQDifXcigLiV4JZ0qBXqEKZi2V3mP
'' SIG '' 2yZWK7Dzp703DNiYdk9WuVLCtp04qYHnbUFcjGnRuSvE
'' SIG '' xnvPnPp44pMadqJpddNQ5EQSviANnqlE0PjlSXcIWiHF
'' SIG '' tM+YlRpUurm8wWkZus8W8oM3NG6wQSbd3lqXTzON1I13
'' SIG '' fXVFoaVYJmoDRd7ZULVQjK9WvUzF4UbFKNOt50MAcN7M
'' SIG '' mJ4ZiQPq1JE3701S88lgIcRWR+3aEUuMMsOI5ljitts+
'' SIG '' +V+wQtaP4xeR0arAVeOGv6wnLEHQmjNKqDbUuXKWfpd5
'' SIG '' OEhfysLcPTLfddY2Z1qJ+Panx+VPNTwAvb6cKmx5Adza
'' SIG '' ROY63jg7B145WPR8czFVoIARyxQMfq68/qTreWWqaNYi
'' SIG '' yjvrmoI1VygWy2nyMpqy0tg6uLFGhmu6F/3Ed2wVbK6r
'' SIG '' r3M66ElGt9V/zLY4wNjsHPW2obhDLN9OTH0eaHDAdwrU
'' SIG '' AuBcYLso/zjlUlrWrBciI0707NMX+1Br/wd3H3GXREHJ
'' SIG '' uEbTbDJ8WC9nR2XlG3O2mflrLAZG70Ee8PBf4NvZrZCA
'' SIG '' RK+AEEGKMIIGrjCCBJagAwIBAgIQBzY3tyRUfNhHrP0o
'' SIG '' ZipeWzANBgkqhkiG9w0BAQsFADBiMQswCQYDVQQGEwJV
'' SIG '' UzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQL
'' SIG '' ExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdp
'' SIG '' Q2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjIwMzIzMDAw
'' SIG '' MDAwWhcNMzcwMzIyMjM1OTU5WjBjMQswCQYDVQQGEwJV
'' SIG '' UzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNV
'' SIG '' BAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
'' SIG '' SEEyNTYgVGltZVN0YW1waW5nIENBMIICIjANBgkqhkiG
'' SIG '' 9w0BAQEFAAOCAg8AMIICCgKCAgEAxoY1BkmzwT1ySVFV
'' SIG '' xyUDxPKRN6mXUaHW0oPRnkyibaCwzIP5WvYRoUQVQl+k
'' SIG '' iPNo+n3znIkLf50fng8zH1ATCyZzlm34V6gCff1DtITa
'' SIG '' EfFzsbPuK4CEiiIY3+vaPcQXf6sZKz5C3GeO6lE98NZW
'' SIG '' 1OcoLevTsbV15x8GZY2UKdPZ7Gnf2ZCHRgB720RBidx8
'' SIG '' ald68Dd5n12sy+iEZLRS8nZH92GDGd1ftFQLIWhuNyG7
'' SIG '' QKxfst5Kfc71ORJn7w6lY2zkpsUdzTYNXNXmG6jBZHRA
'' SIG '' p8ByxbpOH7G1WE15/tePc5OsLDnipUjW8LAxE6lXKZYn
'' SIG '' LvWHpo9OdhVVJnCYJn+gGkcgQ+NDY4B7dW4nJZCYOjgR
'' SIG '' s/b2nuY7W+yB3iIU2YIqx5K/oN7jPqJz+ucfWmyU8lKV
'' SIG '' EStYdEAoq3NDzt9KoRxrOMUp88qqlnNCaJ+2RrOdOqPV
'' SIG '' A+C/8KI8ykLcGEh/FDTP0kyr75s9/g64ZCr6dSgkQe1C
'' SIG '' vwWcZklSUPRR8zZJTYsg0ixXNXkrqPNFYLwjjVj33GHe
'' SIG '' k/45wPmyMKVM1+mYSlg+0wOI/rOP015LdhJRk8mMDDtb
'' SIG '' iiKowSYI+RQQEgN9XyO7ZONj4KbhPvbCdLI/Hgl27Ktd
'' SIG '' RnXiYKNYCQEoAA6EVO7O6V3IXjASvUaetdN2udIOa5kM
'' SIG '' 0jO0zbECAwEAAaOCAV0wggFZMBIGA1UdEwEB/wQIMAYB
'' SIG '' Af8CAQAwHQYDVR0OBBYEFLoW2W1NhS9zKXaaL3WMaiCP
'' SIG '' nshvMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiu
'' SIG '' HA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggr
'' SIG '' BgEFBQcDCDB3BggrBgEFBQcBAQRrMGkwJAYIKwYBBQUH
'' SIG '' MAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggr
'' SIG '' BgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQu
'' SIG '' Y29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQwQwYD
'' SIG '' VR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmww
'' SIG '' IAYDVR0gBBkwFzAIBgZngQwBBAIwCwYJYIZIAYb9bAcB
'' SIG '' MA0GCSqGSIb3DQEBCwUAA4ICAQB9WY7Ak7ZvmKlEIgF+
'' SIG '' ZtbYIULhsBguEE0TzzBTzr8Y+8dQXeJLKftwig2qKWn8
'' SIG '' acHPHQfpPmDI2AvlXFvXbYf6hCAlNDFnzbYSlm/EUExi
'' SIG '' HQwIgqgWvalWzxVzjQEiJc6VaT9Hd/tydBTX/6tPiix6
'' SIG '' q4XNQ1/tYLaqT5Fmniye4Iqs5f2MvGQmh2ySvZ180HAK
'' SIG '' fO+ovHVPulr3qRCyXen/KFSJ8NWKcXZl2szwcqMj+sAn
'' SIG '' gkSumScbqyQeJsG33irr9p6xeZmBo1aGqwpFyd/EjaDn
'' SIG '' mPv7pp1yr8THwcFqcdnGE4AJxLafzYeHJLtPo0m5d2aR
'' SIG '' 8XKc6UsCUqc3fpNTrDsdCEkPlM05et3/JWOZJyw9P2un
'' SIG '' 8WbDQc1PtkCbISFA0LcTJM3cHXg65J6t5TRxktcma+Q4
'' SIG '' c6umAU+9Pzt4rUyt+8SVe+0KXzM5h0F4ejjpnOHdI/0d
'' SIG '' KNPH+ejxmF/7K9h+8kaddSweJywm228Vex4Ziza4k9Tm
'' SIG '' 8heZWcpw8De/mADfIBZPJ/tgZxahZrrdVcA6KYawmKAr
'' SIG '' 7ZVBtzrVFZgxtGIJDwq9gdkT/r+k0fNX2bwE+oLeMt8E
'' SIG '' ifAAzV3C+dAjfwAL5HYCJtnwZXZCpimHCUcr5n8apIUP
'' SIG '' /JiW9lVUKx+A+sDyDivl1vupL0QVSucTDh3bNzgaoSv2
'' SIG '' 7dZ8/DCCBY0wggR1oAMCAQICEA6bGI750C3n79tQ4ghA
'' SIG '' GFowDQYJKoZIhvcNAQEMBQAwZTELMAkGA1UEBhMCVVMx
'' SIG '' FTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQ
'' SIG '' d3d3LmRpZ2ljZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNl
'' SIG '' cnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTIyMDgwMTAw
'' SIG '' MDAwMFoXDTMxMTEwOTIzNTk1OVowYjELMAkGA1UEBhMC
'' SIG '' VVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UE
'' SIG '' CxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMYRGln
'' SIG '' aUNlcnQgVHJ1c3RlZCBSb290IEc0MIICIjANBgkqhkiG
'' SIG '' 9w0BAQEFAAOCAg8AMIICCgKCAgEAv+aQc2jeu+RdSjww
'' SIG '' IjBpM+zCpyUuySE98orYWcLhKac9WKt2ms2uexuEDcQw
'' SIG '' H/MbpDgW61bGl20dq7J58soR0uRf1gU8Ug9SH8aeFaV+
'' SIG '' vp+pVxZZVXKvaJNwwrK6dZlqczKU0RBEEC7fgvMHhOZ0
'' SIG '' O21x4i0MG+4g1ckgHWMpLc7sXk7Ik/ghYZs06wXGXuxb
'' SIG '' Grzryc/NrDRAX7F6Zu53yEioZldXn1RYjgwrt0+nMNlW
'' SIG '' 7sp7XeOtyU9e5TXnMcvak17cjo+A2raRmECQecN4x7ax
'' SIG '' xLVqGDgDEI3Y1DekLgV9iPWCPhCRcKtVgkEy19sEcypu
'' SIG '' kQF8IUzUvK4bA3VdeGbZOjFEmjNAvwjXWkmkwuapoGfd
'' SIG '' pCe8oU85tRFYF/ckXEaPZPfBaYh2mHY9WV1CdoeJl2l6
'' SIG '' SPDgohIbZpp0yt5LHucOY67m1O+SkjqePdwA5EUlibaa
'' SIG '' RBkrfsCUtNJhbesz2cXfSwQAzH0clcOP9yGyshG3u3/y
'' SIG '' 1YxwLEFgqrFjGESVGnZifvaAsPvoZKYz0YkH4b235kOk
'' SIG '' GLimdwHhD5QMIR2yVCkliWzlDlJRR3S+Jqy2QXXeeqxf
'' SIG '' jT/JvNNBERJb5RBQ6zHFynIWIgnffEx1P2PsIV/EIFFr
'' SIG '' b7GrhotPwtZFX50g/KEexcCPorF+CiaZ9eRpL5gdLfXZ
'' SIG '' qbId5RsCAwEAAaOCATowggE2MA8GA1UdEwEB/wQFMAMB
'' SIG '' Af8wHQYDVR0OBBYEFOzX44LScV1kTN8uZz/nupiuHA9P
'' SIG '' MB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6enIZ3zbcgP
'' SIG '' MA4GA1UdDwEB/wQEAwIBhjB5BggrBgEFBQcBAQRtMGsw
'' SIG '' JAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0
'' SIG '' LmNvbTBDBggrBgEFBQcwAoY3aHR0cDovL2NhY2VydHMu
'' SIG '' ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
'' SIG '' dENBLmNydDBFBgNVHR8EPjA8MDqgOKA2hjRodHRwOi8v
'' SIG '' Y3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVk
'' SIG '' SURSb290Q0EuY3JsMBEGA1UdIAQKMAgwBgYEVR0gADAN
'' SIG '' BgkqhkiG9w0BAQwFAAOCAQEAcKC/Q1xV5zhfoKN0Gz22
'' SIG '' Ftf3v1cHvZqsoYcs7IVeqRq7IviHGmlUIu2kiHdtvRoU
'' SIG '' 9BNKei8ttzjv9P+Aufih9/Jy3iS8UgPITtAq3votVs/5
'' SIG '' 9PesMHqai7Je1M/RQ0SbQyHrlnKhSLSZy51PpwYDE3cn
'' SIG '' RNTnf+hZqPC/Lwum6fI0POz3A8eHqNJMQBk1RmppVLC4
'' SIG '' oVaO7KTVPeix3P0c2PR3WlxUjG/voVA9/HYJaISfb8rb
'' SIG '' II01YBwCA8sgsKxYoA5AY8WYIsGyWfVVa88nq2x2zm8j
'' SIG '' LfR+cWojayL/ErhULSd+2DrZ8LaHlv1b0VysGMNNn3O3
'' SIG '' AamfV6peKOK5lDGCA3YwggNyAgEBMHcwYzELMAkGA1UE
'' SIG '' BhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMTsw
'' SIG '' OQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQw
'' SIG '' OTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQQIQBUSv85Sd
'' SIG '' CDmmv9s/X+VhFjANBglghkgBZQMEAgEFAKCB0TAaBgkq
'' SIG '' hkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcN
'' SIG '' AQkFMQ8XDTI0MDEyNjIzNTQ0MlowKwYLKoZIhvcNAQkQ
'' SIG '' AgwxHDAaMBgwFgQUZvArMsLCyQ+CXc6qisnGTxmcz0Aw
'' SIG '' LwYJKoZIhvcNAQkEMSIEIIPD9DmkkcQXFpYftiKL66g1
'' SIG '' fQP2Upc0JkbSnKxL82vtMDcGCyqGSIb3DQEJEAIvMSgw
'' SIG '' JjAkMCIEINL25G3tdCLM0dRAV2hBNm+CitpVmq4zFq9N
'' SIG '' GprUDHgoMA0GCSqGSIb3DQEBAQUABIICAFFBtuUB2yTm
'' SIG '' Ugqy2kAPHMuY/fD+zM9i6aVP6O8NZXuWQ+25Mrc2Y+AX
'' SIG '' F++W+o2yZCkslwytXS0EhoOBKXio9wbMAZ/dxfsxsMux
'' SIG '' PKrcwBD1KZ22WG7oCWqmO+lx2uWgximtW+9OERhAYe5Z
'' SIG '' 5ERKIEiTXZQm4rzluF6wtXkt63LlGDLZKe2oUpVfgIGK
'' SIG '' cyqzsMbWHsO0DB+lFrDpmwF7zCGetaZ37Z5+NmuvsWQW
'' SIG '' vlo2AdfHNgJxu3SI3zDb6Y1g1B+dADXEfUm1u8IW4NhH
'' SIG '' iFm9GAdLuw2g1i6Ok1ohAP+Kl8zsrGW9CL3cWG5RNNBF
'' SIG '' S19XMf7nyP8tRHvBBvP4adlwSez7fPuq2F8GgCPMpp+5
'' SIG '' w5cQX879FxEFh9D938nWSv0i+8dZYWCgUbMqoZ+A+zCb
'' SIG '' 9lwN9pB8FqsFSZu+gHglptyn6gVb0FxgCAwklzY4jhxj
'' SIG '' EApMzviL3x7fcvkPsMZ9aujKtaY+ly/UpRdpVunyS5px
'' SIG '' VvHrgvHGWCAhAhO6x5GH7Tc0jvhigz/iPwgnybh4chKA
'' SIG '' 52WO1JuG/Qju99K2Iz0Fk2fP/OLgrP49QIFjy4SIWG8I
'' SIG '' Qn4fhPXJBRC1D3ifys/JuE0/q/Q1BdS9f1xM9m2uKgpt
'' SIG '' Vt7p8EwaiwGWtdu9sP2CQ5IuJdv1Y5VqLo7DD3038Ys7
'' SIG '' T2OQOV3iadfq
'' SIG '' End signature block
